/*
 * Copyright (C) 2012-2025 Japan Smartphone Security Association
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.jssec.notification.visibilityPrivate;

import static android.app.PendingIntent.FLAG_MUTABLE;

import android.app.Activity;
import android.app.Notification;
import android.app.NotificationChannel;
import android.app.NotificationManager;
import android.app.PendingIntent;
import android.app.Person;
import android.app.RemoteInput;
import android.content.Context;
import android.content.Intent;
import android.graphics.drawable.Icon;
import android.net.Uri;
import android.os.Build;
import android.os.Bundle;
import android.view.View;

public class VisibilityPrivateNotificationActivity extends Activity {
    /**
     * Display a private Notification
     */
    private final int mNotificationId = 0;

    public static final String DEFAULT_CHANNEL = "default_channel";
    public static final String SENDER_NAME = "Sender Name";

    public static final String REMOTE_REPLY = "remote_reply";
    public static final String reply_choices[] = {"choice1", "choice2", "choice3"};
    public static final String REPLY_LABEL = "input reply";

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.activity_main);

        if (Build.VERSION.SDK_INT >= 26) {
            // Channel is required for Notification from api level 26	
            NotificationChannel default_channel =
                new NotificationChannel(DEFAULT_CHANNEL,
                    getString(R.string.notification_channel_default),
                    NotificationManager.IMPORTANCE_DEFAULT);

            NotificationManager notificationManager = (NotificationManager) this.getSystemService(Context.NOTIFICATION_SERVICE);
            notificationManager.createNotificationChannel(default_channel);
        }
    }

    public void onSendNotificationClick(View view) {

        // *** POINT 1 *** When preparing a Notification that includes private
        // information, prepare an additional Noficiation for public display
        // (displayed when the screen is locked).
        Notification.Builder publicNotificationBuilder;
        if (Build.VERSION.SDK_INT >= 26) {
            publicNotificationBuilder =
                new Notification.Builder(this, DEFAULT_CHANNEL)
                        .setContentTitle("Notification : Public");
        } else {
            publicNotificationBuilder =
                new Notification.Builder(this)
                        .setContentTitle("Notification : Public");
        }

        if (Build.VERSION.SDK_INT >= 21)
            publicNotificationBuilder
                .setVisibility(Notification.VISIBILITY_PUBLIC);
        // *** POINT 2 *** Do not include private information in Notifications
        // prepared for public display (displayed when the screen is locked).
        publicNotificationBuilder.setContentText("Visibility Public : Omitting sensitive data.");
        publicNotificationBuilder.setSmallIcon(R.drawable.ic_launcher);
        Notification publicNotification = publicNotificationBuilder.build();

        // Construct a Notification that includes private information.
        Notification.Builder privateNotificationBuilder;
        if (Build.VERSION.SDK_INT >= 26) {
            privateNotificationBuilder =
                    new Notification.Builder(this, DEFAULT_CHANNEL)
                            .setContentTitle("Notification : Private");
        } else {
            privateNotificationBuilder =
                    new Notification.Builder(this)
                            .setContentTitle("Notification : Private");
        }

        // *** POINT 3 *** Explicitly set Visibility to Private when creating
        // Notifications.
        if (Build.VERSION.SDK_INT >= 21)
            privateNotificationBuilder
                .setVisibility(Notification.VISIBILITY_PRIVATE);
        // *** POINT 4 *** When Visibility is set to Private, Notifications may
        // contain private information.
        privateNotificationBuilder
            .setContentText("Visibility Private : Including user info.");
        privateNotificationBuilder.setSmallIcon(R.drawable.ic_launcher);
        // When creating a Notification with Visibility=Private, we also create
        // and register a separate Notification with Visibility=Public for public
        // display.
        if (Build.VERSION.SDK_INT >= 21)
            privateNotificationBuilder
                .setPublicVersion(publicNotification);

        if (Build.VERSION.SDK_INT >= 28) {
            // Display resource sample_picture.png in Notification by
            // Notification.MessagingStyle.Message.setData()
            int resourceId = R.drawable.sample_picture_small;
            Uri imageUri = Uri.parse("android.resource://" +
                                     getApplicationContext().getPackageName() +
                                     "/" + resourceId);

            Person sender = new Person.Builder()
                    .setName(SENDER_NAME)
                    .setIcon(null)
                    .build();

            // Prepare Notification.MessagingStyle.Message and set the image with
            // setData()
            Notification.MessagingStyle.Message message =
                new Notification.MessagingStyle
                        .Message("Sample Picture", 0, sender)
                        .setData("image/png", imageUri);

            // Prepare Notification.MessagingStyle and set
            // Notification.MessagingStyle.Message that sets the image
            Notification.MessagingStyle message_style =
                    new Notification.MessagingStyle(sender)
                    .addMessage(message);

            // Set Notification.MessagingStyle to Notification
            privateNotificationBuilder.setStyle(message_style);
        }

        if (Build.VERSION.SDK_INT >= 28) {
            // Display reply options in Notification by
            // RemoteInput.Builder.setChoices()
            Intent intent =
                new Intent( getApplicationContext(), NotificationReceiver.class);

            PendingIntent pendingIntent =
                PendingIntent.getBroadcast(this, 0, intent, 0);

            RemoteInput remoteInput = new RemoteInput.Builder(REMOTE_REPLY)
                    .setLabel(REPLY_LABEL)
                    .setChoices(reply_choices)
                    .build();

            Icon icon = Icon.createWithResource(this, R.drawable.ic_launcher);

            Notification.Action actionReply =
                new Notification.Action.Builder(icon, REPLY_LABEL, pendingIntent)
                    .addRemoteInput(remoteInput)
                    .setSemanticAction(Notification.Action.SEMANTIC_ACTION_REPLY)
                    .build();

            privateNotificationBuilder.addAction(actionReply);
        }

        Notification privateNotification = privateNotificationBuilder.build();
        //Although not implemented in this sample code, in many cases
        //Notifications will use setContentIntent(PendingIntent intent)
        //to ensure that an Intent is transmission when Notification
        //is clicked. In this case, it is necessary to take steps--depending
        //on the type of component being called--to ensure that the Intent
        //in question is called by safe methods (for example, by explicitly
        //using Intent). For information on safe methods for calling various
        //types of component, see the following sections.
        //4.1. Creating and using Activities
        //4.2. Sending and receiving Broadcasts
        //4.4. Creating and using Services

        NotificationManager notificationManager =
                (NotificationManager) this
                .getSystemService(Context.NOTIFICATION_SERVICE);
        notificationManager.notify(mNotificationId, privateNotification);
    }
}
